import React from "react";
import Image from "next/image";

type StripPhase = "idle" | "spinning" | "stopping" | "showWinner";

type RollStripProps = {
stripViewportRef: React.RefObject<HTMLDivElement | null>;
  stripTrackRef: React.RefObject<HTMLDivElement | null>

  stripPhase: StripPhase;
  stripOffset: number;
  stopTransition?: string;

  stripTrack: number[]; // array of img indexes
  stripImages: string[]; // array of image srcs

  winnerTrackIndex: number | null;
  showWinnerEffects: boolean;
};

export default function RollStrip({
  stripViewportRef,
  stripTrackRef,
  stripPhase,
  stripOffset,
  stopTransition,
  stripTrack,
  stripImages,
  winnerTrackIndex,
  showWinnerEffects,
}: RollStripProps) {
  return (
    <div className="relative mt-5 bg-black">
      {/* viewport */}
      <div ref={stripViewportRef} className="relative py-3 pb-8 overflow-hidden">
        {/* spinning track */}
        <div
          ref={stripTrackRef}
          className="flex items-center justify-center gap-2 scrollbar-hide will-change-transform"
          style={
            stripPhase === "stopping" || stripPhase === "showWinner"
              ? {
                  transform: `translateX(${stripOffset}px)`,
                  transition:
                    stopTransition ||
                    "transform 0.8s cubic-bezier(0.22, 0.61, 0.36, 1)",
                }
              : {}
          }
        >
          {stripTrack.map((imgIdx, idx) => {
            const isWinner =
              stripPhase === "showWinner" && winnerTrackIndex === idx;

            const imageSrc = stripImages[imgIdx];

            return (
              <div
                key={idx}
                data-strip-index={idx}
                className={`relative shrink-0 rounded-xl bg-[#161E2A] w-18 h-18 md:w-20 md:h-20 2xl:w-24 2xl:h-24 flex items-center justify-center transition-all duration-300
                  ${
                    isWinner && showWinnerEffects
                      ? "border border-[#96B56A] winnerShadow winner-border"
                      : ""
                  }
                `}
              >
                <div className="absolute inset-1 rounded-2xl" />

                <div className="relative flex items-center justify-center">
                  <Image
                    src={imageSrc}
                    alt="icon"
                    width={40}
                    height={40}
                    className="w-12 h-12 md:w-14 md:h-14 2xl:w-16 2xl:h-16 object-contain"
                  />
                </div>

                {/* Winner arrow */}
                {isWinner && showWinnerEffects && (
                  <img
                    src="/images/roll/winner.svg"
                    alt="winner"
                    className="pointer-events-none absolute -bottom-7 left-1/2 -translate-x-1/2 w-14 md:w-20"
                  />
                )}
              </div>
            );
          })}
        </div>
      </div>

      {/* Edge gradients stay the same */}
      <div className="pointer-events-none absolute inset-y-0 z-10 left-0 w-40 bg-gradient-to-r from-[#12171C] via-[#12171C]/80 to-transparent" />
      <div className="pointer-events-none absolute inset-y-0 z-10 right-0 w-40 bg-gradient-to-l from-[#12171C] via-[#12171C]/80 to-transparent" />
    </div>
  );
}
