const ranges = [
    { minRange: 1, maxRange: 2, tag: "Code Card" },
    { minRange: 1, maxRange: 9, tag: "Energy" },
    { minRange: 1, maxRange: 42, tag: "Common" },
    { minRange: 1, maxRange: 42, tag: "Common" },
    { minRange: 1, maxRange: 42, tag: "Common" },
    { minRange: 1, maxRange: 42, tag: "Common" },
    { minRange: 1, maxRange: 42, tag: "Common" },
    { minRange: 1, maxRange: 51, tag: "Uncommon" },
    { minRange: 1, maxRange: 51, tag: "Uncommon" },
    { minRange: 1, maxRange: 51, tag: "Uncommon" },
    { minRange: 1, maxRange: 100, tag: "reverseHoloCategory" },
    { minRange: 1, maxRange: 42, tag: "Common , Reverse Holofoil" },
    { minRange: 1, maxRange: 51, tag: "Uncommon , Reverse Holofoil" },
    { minRange: 1, maxRange: 19, tag: "Rare , Reverse Holofoil" },
    { minRange: 1, maxRange: 20, tag: "Holo Rare , Reverse Holofoil" },
    { minRange: 1, maxRange: 10000, tag: "rareCategory" },
    { minRange: 1, maxRange: 19, tag: "Rare" },
    { minRange: 1, maxRange: 20, tag: "Holo Rare" },
    { minRange: 1, maxRange: 18, tag: "V" },
    { minRange: 1, maxRange: 15, tag: "Vmax" },
    { minRange: 1, maxRange: 29, tag: "Full Art" },
    { minRange: 1, maxRange: 9, tag: "Alt Art V" },
    { minRange: 1, maxRange: 6, tag: "Alt Art Vmax" },
    { minRange: 1, maxRange: 16, tag: "Rainbow Rare" },
    { minRange: 1, maxRange: 12, tag: "Gold Rare" }
];

function incrementNonce(nonce, offset) {
  let seconds = parseInt(nonce.slice(0, 2), 10);
  let minutes = parseInt(nonce.slice(2, 4), 10);
  let hours = parseInt(nonce.slice(4, 6), 10);
  let day = parseInt(nonce.slice(6, 8), 10);
  let month = parseInt(nonce.slice(8, 10), 10);
  let year = parseInt(nonce.slice(10, 14), 10);

  seconds += offset;
  while (seconds > 59) {
    seconds -= 60;
    minutes += 1;
    if (minutes > 59) {
      minutes = 0;
      hours += 1;
      if (hours > 23) {
        hours = 0;
        day += 1;
        const daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        if (day > daysInMonth[month - 1]) {
          day = 1;
          month += 1;
          if (month > 12) {
            month = 1;
            year += 1;
          }
        }
      }
    }
  }

  return (
    seconds.toString().padStart(2, '0') +
    minutes.toString().padStart(2, '0') +
    hours.toString().padStart(2, '0') +
    day.toString().padStart(2, '0') +
    month.toString().padStart(2, '0') +
    year.toString().padStart(4, '0')
  );
}

async function generateHmacSha512(key, message) {
  const keyBuffer = new TextEncoder().encode(key);
  const msgBuffer = new TextEncoder().encode(message);
  const cryptoKey = await crypto.subtle.importKey(
    "raw",
    keyBuffer,
    { name: "HMAC", hash: "SHA-512" },
    false,
    ["sign"]
  );
  const signature = await crypto.subtle.sign("HMAC", cryptoKey, msgBuffer);
  return new Uint8Array(signature);
}

async function generateRandomNumbers(ranges, serverSeed, clientSeed, nonce) {
  const randomNumbers = [];
  const numbersPerDigest = 16;
  const digestsNeeded = Math.ceil(ranges.length / numbersPerDigest);

  for (let digestIndex = 0; digestIndex < digestsNeeded; digestIndex++) {
    const message = `${clientSeed}:${nonce}:${digestIndex}`;
    const digest = await generateHmacSha512(serverSeed, message);
    let offset = 0;

    for (let i = 0; i < numbersPerDigest && randomNumbers.length < ranges.length; i++) {
      const range = ranges[randomNumbers.length];
      const bytes = digest.slice(offset, offset + 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const num = view.getUint32(0, false); // Big-endian
      const rand = Math.floor((num / 0x100000000) * (range.maxRange - range.minRange + 1)) + range.minRange;
      randomNumbers.push(rand);
      offset += 4;
    }
  }

  return randomNumbers;
}

async function mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce) {
  const seenValues = {};
  const taggedResults = [];

  for (let index = 0; index < randomNumbers.length; index++) {
    if (!ranges[index]) {
      console.error(`No range found for index ${index}`);
      continue;
    }

    const { tag = "UNKNOWN", minRange, maxRange } = ranges[index];
    let finalNumber = randomNumbers[index];
    let nonceOffset = 0;

    if (!seenValues[tag]) {
      seenValues[tag] = new Set();
    }

    while (seenValues[tag].has(finalNumber.toString())) {
      nonceOffset++;
      const newNonce = incrementNonce(nonce, nonceOffset);
      const message = `${clientSeed}:${newNonce}`;
      const digest = await generateHmacSha512(serverSeed, message);
      const bytes = digest.slice(0, 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const newNum = view.getUint32(0, false); // Big-endian
      finalNumber = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
    }

    seenValues[tag].add(finalNumber.toString());
    taggedResults.push({ value: finalNumber.toString(), tag });
  }

  return taggedResults;
}

async function filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce) {
  let filteredResults = [];
  let nonceOffset = 0;

  const codeCard = taggedResults.find(item => item.tag === "Code Card");
  if (codeCard) filteredResults.push(codeCard);

  const energyCard = taggedResults.find(item => item.tag === "Energy");
  if (energyCard) filteredResults.push(energyCard);

  const commonCards = taggedResults.filter(item => item.tag === "Common").slice(0, 5);
  filteredResults.push(...commonCards);

  const uncommonCards = taggedResults.filter(item => item.tag === "Uncommon").slice(0, 3);
  filteredResults.push(...uncommonCards);

  const reverseHoloCategoryValue = taggedResults.find(item => item.tag === "reverseHoloCategory")?.value;
  if (reverseHoloCategoryValue) {
    const reverseHoloCategoryNum = parseInt(reverseHoloCategoryValue, 10);
    if (reverseHoloCategoryNum >= 1 && reverseHoloCategoryNum <= 60) {
      const commonReverseHolo = taggedResults.find(item => item.tag === "Common , Reverse Holofoil");
      if (commonReverseHolo) filteredResults.push(commonReverseHolo);
    } else if (reverseHoloCategoryNum >= 61 && reverseHoloCategoryNum <= 90) {
      const uncommonReverseHolo = taggedResults.find(item => item.tag === "Uncommon , Reverse Holofoil");
      if (uncommonReverseHolo) filteredResults.push(uncommonReverseHolo);
    } else if (reverseHoloCategoryNum >= 91 && reverseHoloCategoryNum <= 99) {
      const rareReverseHolo = taggedResults.find(item => item.tag === "Rare , Reverse Holofoil");
      if (rareReverseHolo) filteredResults.push(rareReverseHolo);
    } else if (reverseHoloCategoryNum === 100) {
      const holoRareReverseHolo = taggedResults.find(item => item.tag === "Holo Rare , Reverse Holofoil");
      if (holoRareReverseHolo) filteredResults.push(holoRareReverseHolo);
    }
  }

  const rareCategoryValue = taggedResults.find(item => item.tag === "rareCategory")?.value;
  if (rareCategoryValue) {
    const rareCategoryNum = parseInt(rareCategoryValue, 10);
    let rareItem = null;
    let rareTag = null;
    if (rareCategoryNum >= 1 && rareCategoryNum <= 5889) {
      rareItem = taggedResults.find(item => item.tag === "Rare");
      rareTag = "Rare";
    } else if (rareCategoryNum >= 5890 && rareCategoryNum <= 8834) {
      rareItem = taggedResults.find(item => item.tag === "Holo Rare");
      rareTag = "Holo Rare";
    } else if (rareCategoryNum >= 8835 && rareCategoryNum <= 9013) {
      rareItem = taggedResults.find(item => item.tag === "V");
      rareTag = "V";
    } else if (rareCategoryNum >= 9014 && rareCategoryNum <= 9569) {
      rareItem = taggedResults.find(item => item.tag === "Vmax");
      rareTag = "Vmax";
    } else if (rareCategoryNum >= 9570 && rareCategoryNum <= 9620) {
      rareItem = taggedResults.find(item => item.tag === "Full Art");
      rareTag = "Full Art";
    } else if (rareCategoryNum >= 9621 && rareCategoryNum <= 9742) {
      rareItem = taggedResults.find(item => item.tag === "Rainbow Rare");
      rareTag = "Rainbow Rare";
    } else if (rareCategoryNum >= 9743 && rareCategoryNum <= 9843) {
      rareItem = taggedResults.find(item => item.tag === "Gold Rare");
      rareTag = "Gold Rare";
    } else if (rareCategoryNum >= 9844 && rareCategoryNum <= 9965) {
      rareItem = taggedResults.find(item => item.tag === "Alt Art V");
      rareTag = "Alt Art V";
    } else if (rareCategoryNum >= 9966 && rareCategoryNum <= 10000) {
      rareItem = taggedResults.find(item => item.tag === "Alt Art Vmax");
      rareTag = "Alt Art Vmax";
    }

    if (rareItem) filteredResults.push(rareItem);
  }

  return filteredResults;
}

async function setData(serverSeed, clientSeed, nonce) {
  const randomNumbers = await generateRandomNumbers(ranges, serverSeed, clientSeed, nonce);
  const taggedResults = await mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce);
  const filteredResults = await filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce);

  return {
    success: true,
    result: filteredResults,
    nonce,
    serverSeed,
    clientSeed
  };
}
export { setData, ranges};
