"use client";

const codeFormula = `const rnd = Math.floor((num / 0x100000000) * (range.maxRange - range.minRange + 1)) + range.minRange;`;

const jsCode = `// CONVERT DIGEST BYTES TO SCALED NUMBERS FOR PACK RANGES

function generateScaledNumbers({ serverSeed, clientSeed, nonce, ranges }) {
  const crypto = require("crypto");
  const scaledNumbers = [];
  const numbersPerDigest = 16;
  const digestsNeeded = Math.ceil(ranges.length / numbersPerDigest);

  for (let digestIndex = 0; digestIndex < digestsNeeded; digestIndex++) {
    const hmac = crypto.createHmac("sha512", serverSeed);
    hmac.update(\`\${clientSeed}:\${nonce}:\${digestIndex}\`);
    const digest = hmac.digest();
    let offset = 0;

    for (
      let i = 0;
      i < numbersPerDigest && scaledNumbers.length < ranges.length;
      i++
    ) {
      const range = ranges[scaledNumbers.length];
      const bytes = digest.slice(offset, offset + 4);
      const num = bytes.readUInt32BE(0);
      const scaled =
        Math.floor(
          (num / 0x100000000) * (range.maxRange - range.minRange + 1)
        ) + range.minRange;

      scaledNumbers.push(scaled);
      offset += 4;
    }
  }

  return scaledNumbers;
}
`;

export const ImplementContent = () => {
  return (
    <section className="w-full font-onest text-white pb-12">
      <div className="container mx-auto ">
        {/* Heading */}
        <h2 className="text-sm italic font-sofia md:text-2xl font-extrabold tracking-tight uppercase mb-3">
          INVENTORY
        </h2>

        {/* Description */}
        <p className="text-xs md:text-sm font-onest text-[#DCDCDC] leading-relaxed mb-3 max-w-4xl">
          The output of the random number generation process (from
          <span className="font-mono"> generateRandomNumbers</span>) is derived
          from a 64-byte <span className="font-mono">HMAC_SHA512</span> digest.
          As described in the digest index implementation, we use 4 bytes of
          data to generate a single random number for each rarity category. Each
          set of 4 bytes is converted to an unsigned 32-bit integer using
          big-endian byte order (
          <span className="font-mono">bytes.readUInt32BE(0)</span>), which is
          then scaled to produce a number within the specified range for that
          category (e.g., 1–72 for Common, 1–13 for rare). The scaling formula
          is:
        </p>

        {/* One-line formula block */}
        <div className="mt-2 mb-4 bg-[var(--color-naviBlue)] font-sofia px-4 py-3 overflow-x-auto">
          <pre className="font-mono text-[10px] md:text-xs text-slate-100 whitespace-pre">
            {codeFormula}
          </pre>
        </div>

        <p className="text-xs font-onest md:text-sm text-[#DCDCDC]  leading-relaxed mb-6 max-w-4xl">
          This ensures high precision and uniformity in the randomness used for
          pack openings. The following code snippet, adapted from the provided
          logic, illustrates how bytes are converted to scaled numbers for pack
          ranges:
        </p>

        {/* Large code block */}
        <div className="bg-[var(--color-naviBlue)]  px-4 md:px-6 py-4 md:py-6 overflow-x-auto">
          <pre className="font-mono text-[10px] font-onest md:text-xs leading-relaxed text-slate-100 whitespace-pre">
            {jsCode}
          </pre>
        </div>
      </div>
    </section>
  );
};
