import { NextResponse } from "next/server";
import crypto from "crypto";

function safeEqual(a: string, b: string) {
  const ab = Buffer.from(a);
  const bb = Buffer.from(b);
  if (ab.length !== bb.length) return false;
  return crypto.timingSafeEqual(ab, bb);
}

export async function POST(req: Request) {
  const body = await req.json();

  const callbackSecret = process.env.B2BINPAY_CALLBACK_SECRET!;
  const metaTime = body?.meta?.time ?? "";
  const sign = body?.meta?.sign ?? "";

  // The docs define the message as concatenation of:
  // transfer.status + transfer.amount + deposit.tracking_id + meta.time
  // (Make sure you match their exact formatting expectations.)
  const transfer = body?.included?.find((x: any) => x.type === "transfer")?.attributes;
  const deposit = body?.data?.attributes;

  const message =
    String(transfer?.status ?? "") +
    String(transfer?.amount ?? "") +
    String(deposit?.tracking_id ?? "") +
    String(metaTime);

  const expected = crypto.createHmac("sha256", callbackSecret).update(message).digest("hex");

  if (!safeEqual(expected, String(sign))) {
    return new NextResponse("Invalid signature", { status: 400 });
  }

  // ✅ Verified. Now update your DB:
  // - find order by tracking_id
  // - mark paid/confirmed based on status, amounts, confirmations, etc.

  return NextResponse.json({ ok: true }); // they want HTTP 200 with no body; JSON 200 is usually fine, but you can also return empty 200.
}
